// ============================================
// BASEWUMP - Standalone Version
// Live Base Chain Token Visualizer
// Works with VS Code Live Server (port 5500)
// ============================================

// Configuration
const CONFIG = {
    BASESCAN_API_KEY: "7QW1GDFJ5BPYIT54RA6DI2VQ4PTJ7AA3QY",
    ROUTER_ADDRESS: "0x4752ba5DBc23f44D87826276BF6Fd6b1C372aD24",
    BASESCAN_API_URL: "https://api.basescan.org/api",
    POLL_INTERVAL: 5000, // 5 seconds
    MAX_TOKENS: 50,
    ETH_PRICE_USD: 3000, // Approximate for mcap estimation
};

// Function signatures for Uniswap V2
const ADD_LIQ_ETH_SIG = "0xf305d719";
const ADD_LIQ_SIG = "0xe8e33700";

// Known quote tokens to filter out
const QUOTE_TOKENS = new Set([
    "0x4200000000000000000000000000000000000006", // WETH
    "0x833589fcd6edb6e08f4c7c32d4f71b54bda02913", // USDC
    "0x50c5725949a6f0c72e6c4a641f24049a917db0cb", // DAI
]);

// Available images for tokens
const TOKEN_IMAGES = [
    "images/doge.png",
    "images/cat.png",
    "images/rocket.png",
    "images/logo.png"
];

// State
let tokens = [];
let scale = 1;
let position = { x: 0, y: 0 };
let isDragging = false;
let lastMousePos = { x: 0, y: 0 };
let usingRealData = false;

// DOM Elements
const bubblesContainer = document.getElementById("bubbles-container");
const feedList = document.getElementById("feed-list");
const statusText = document.getElementById("status-text");
const pulseDot = document.querySelector(".pulse-dot");
const tooltip = document.getElementById("tooltip");
const toast = document.getElementById("toast");
const toastMessage = document.getElementById("toast-message");

// ============================================
// UTILITY FUNCTIONS
// ============================================

function randomChoice(arr) {
    return arr[Math.floor(Math.random() * arr.length)];
}

function randomRange(min, max) {
    return Math.random() * (max - min) + min;
}

function formatMcap(mcap) {
    if (mcap >= 1000000) return `$${(mcap / 1000000).toFixed(1)}M`;
    if (mcap >= 1000) return `$${(mcap / 1000).toFixed(1)}k`;
    return `$${mcap.toFixed(0)}`;
}

function truncateAddress(addr) {
    if (!addr || addr.length < 10) return addr;
    return addr.substring(0, 6) + "..." + addr.substring(addr.length - 4);
}

// Decode ABI-encoded string (for token name/symbol)
function decodeString(hex) {
    try {
        const raw = hex.replace("0x", "");
        let str = "";
        for (let i = 0; i < raw.length; i += 2) {
            const code = parseInt(raw.substr(i, 2), 16);
            if (code > 31 && code < 127) {
                str += String.fromCharCode(code);
            }
        }
        return str.replace(/[^a-zA-Z0-9 ]/g, "").trim();
    } catch (e) {
        return "Unknown";
    }
}

// ============================================
// API FUNCTIONS
// ============================================

async function fetchTokenDetails(address) {
    try {
        // Fetch Symbol (0x95d89b41)
        const symbolRes = await fetch(
            `${CONFIG.BASESCAN_API_URL}?module=proxy&action=eth_call&to=${address}&data=0x95d89b41&apikey=${CONFIG.BASESCAN_API_KEY}`
        );
        const symbolData = await symbolRes.json();

        // Fetch Name (0x06fdde03)
        const nameRes = await fetch(
            `${CONFIG.BASESCAN_API_URL}?module=proxy&action=eth_call&to=${address}&data=0x06fdde03&apikey=${CONFIG.BASESCAN_API_KEY}`
        );
        const nameData = await nameRes.json();

        return {
            symbol: symbolData.result && symbolData.result !== "0x" ? decodeString(symbolData.result) : "UNK",
            name: nameData.result && nameData.result !== "0x" ? decodeString(nameData.result) : "Unknown Token"
        };
    } catch (e) {
        return { symbol: "UNK", name: "Unknown Token" };
    }
}

async function fetchRouterActivity() {
    try {
        const response = await fetch(
            `${CONFIG.BASESCAN_API_URL}?module=account&action=txlist&address=${CONFIG.ROUTER_ADDRESS}&startblock=0&endblock=99999999&page=1&offset=50&sort=desc&apikey=${CONFIG.BASESCAN_API_KEY}`
        );
        const data = await response.json();

        if (data.status === "1" && Array.isArray(data.result)) {
            const newLaunches = [];

            // Filter for addLiquidity transactions
            const liquidityTxs = data.result.filter(tx =>
                tx.input && (tx.input.startsWith(ADD_LIQ_ETH_SIG) || tx.input.startsWith(ADD_LIQ_SIG))
            );

            // Process up to 10 transactions
            for (const tx of liquidityTxs.slice(0, 10)) {
                let tokenAddress = "";

                try {
                    if (tx.input.startsWith(ADD_LIQ_ETH_SIG)) {
                        // addLiquidityETH(token, ...) -> Token is always first param
                        tokenAddress = "0x" + tx.input.substring(34, 74);
                    } else if (tx.input.startsWith(ADD_LIQ_SIG)) {
                        // addLiquidity(tokenA, tokenB, ...)
                        const tokenA = "0x" + tx.input.substring(34, 74).toLowerCase();
                        const tokenB = "0x" + tx.input.substring(98, 138).toLowerCase();

                        // Pick the one that is NOT a quote token
                        if (QUOTE_TOKENS.has(tokenA)) {
                            tokenAddress = tokenB;
                        } else if (QUOTE_TOKENS.has(tokenB)) {
                            tokenAddress = tokenA;
                        } else {
                            tokenAddress = tokenA;
                        }
                    }
                } catch (err) {
                    console.error("Error parsing tx input", err);
                    continue;
                }

                if (!tokenAddress || !tokenAddress.match(/^0x[a-fA-F0-9]{40}$/)) continue;

                // Check if we already have this token
                if (tokens.find(t => t.id === tx.hash)) continue;

                const details = await fetchTokenDetails(tokenAddress);

                const ethValue = parseFloat(tx.value) / 1e18;
                const estimatedMcap = ethValue > 0 ? ethValue * CONFIG.ETH_PRICE_USD * 2 : 5000;

                newLaunches.push({
                    id: tx.hash,
                    name: details.name.substring(0, 20),
                    ticker: details.symbol.substring(0, 8),
                    ca: tokenAddress,
                    devWallet: tx.from,
                    mcap: estimatedMcap,
                    createdAt: parseInt(tx.timeStamp) * 1000,
                    isSniper: tx.isError === "1",
                    holders: 1,
                    image: randomChoice(TOKEN_IMAGES),
                    change24h: randomRange(-20, 100),
                    x: randomRange(-800, 800),
                    y: randomRange(-500, 500),
                    vx: randomRange(-0.3, 0.3),
                    vy: randomRange(-0.3, 0.3),
                });
            }

            return newLaunches;
        }
        return [];
    } catch (error) {
        console.error("Failed to fetch Basescan data:", error);
        return [];
    }
}

// ============================================
// RENDERING FUNCTIONS
// ============================================

function renderBubbles() {
    bubblesContainer.innerHTML = "";

    tokens.forEach(token => {
        const size = Math.max(60, Math.log10(Math.max(token.mcap, 1000)) * 35);
        const isPositive = token.change24h >= 0;

        const bubble = document.createElement("div");
        bubble.className = `bubble ${isPositive ? "" : "negative"}`;
        bubble.style.cssText = `
            width: ${size}px;
            height: ${size}px;
            left: ${token.x}px;
            top: ${token.y}px;
            box-shadow: 0 0 ${size / 3}px ${isPositive ? "rgba(34, 197, 94, 0.5)" : "rgba(239, 68, 68, 0.5)"};
        `;
        bubble.dataset.tokenId = token.id;

        bubble.innerHTML = `
            <img class="bubble-img" src="${token.image}" alt="${token.name}">
            <div class="bubble-overlay">
                <span class="bubble-ticker">${token.ticker}</span>
                <span class="bubble-change ${isPositive ? "positive" : "negative"}">
                    ${isPositive ? "+" : ""}${token.change24h.toFixed(1)}%
                </span>
            </div>
        `;

        // Events
        bubble.addEventListener("click", () => copyCA(token));
        bubble.addEventListener("mouseenter", (e) => showTooltip(e, token));
        bubble.addEventListener("mouseleave", hideTooltip);
        bubble.addEventListener("mousemove", (e) => moveTooltip(e));

        bubblesContainer.appendChild(bubble);
    });
}

function renderFeed() {
    feedList.innerHTML = "";

    const recentTokens = [...tokens].reverse().slice(0, 5);

    recentTokens.forEach(token => {
        const isPositive = token.change24h >= 0;

        const item = document.createElement("div");
        item.className = "feed-item";
        item.innerHTML = `
            <div class="feed-item-left">
                <img class="feed-item-img" src="${token.image}" alt="${token.name}">
                <div>
                    <div class="feed-item-name">${token.ticker}</div>
                    <div class="feed-item-holders">${token.holders} Holders</div>
                </div>
            </div>
            <div class="feed-item-right">
                <div class="feed-item-mcap">${formatMcap(token.mcap)}</div>
                <div class="feed-item-change ${isPositive ? "positive" : "negative"}">
                    ${isPositive ? "+" : ""}${token.change24h.toFixed(1)}%
                </div>
            </div>
        `;

        item.addEventListener("click", () => copyCA(token));
        feedList.appendChild(item);
    });
}

function updateCanvasTransform() {
    bubblesContainer.style.transform = `translate(${position.x}px, ${position.y}px) scale(${scale})`;
}

// ============================================
// TOOLTIP FUNCTIONS
// ============================================

function showTooltip(e, token) {
    document.getElementById("tooltip-name").textContent = token.name;
    document.getElementById("tooltip-ticker").textContent = token.ticker;
    document.getElementById("tooltip-mcap").textContent = formatMcap(token.mcap);
    document.getElementById("tooltip-holders").textContent = token.holders;
    document.getElementById("tooltip-dev").textContent = truncateAddress(token.devWallet);
    document.getElementById("tooltip-dev").title = token.devWallet;
    document.getElementById("tooltip-ca").textContent = truncateAddress(token.ca);
    document.getElementById("tooltip-ca").title = token.ca;
    document.getElementById("tooltip-basescan").href = `https://basescan.org/token/${token.ca}`;

    const sniperWarning = document.getElementById("sniper-warning");
    if (token.isSniper) {
        sniperWarning.classList.remove("hidden");
    } else {
        sniperWarning.classList.add("hidden");
    }

    tooltip.classList.remove("hidden");
    moveTooltip(e);
}

function hideTooltip() {
    tooltip.classList.add("hidden");
}

function moveTooltip(e) {
    const x = e.clientX + 15;
    const y = e.clientY + 15;

    // Keep tooltip on screen
    const tooltipRect = tooltip.getBoundingClientRect();
    const maxX = window.innerWidth - tooltipRect.width - 10;
    const maxY = window.innerHeight - tooltipRect.height - 10;

    tooltip.style.left = `${Math.min(x, maxX)}px`;
    tooltip.style.top = `${Math.min(y, maxY)}px`;
}

// ============================================
// COPY & TOAST
// ============================================

function copyCA(token) {
    navigator.clipboard.writeText(token.ca).then(() => {
        showToast(`${token.ticker} CA copied!`);
    }).catch(err => {
        console.error("Failed to copy:", err);
        showToast("Failed to copy");
    });
}

function showToast(message) {
    toastMessage.textContent = message;
    toast.classList.remove("hidden");

    setTimeout(() => {
        toast.classList.add("hidden");
    }, 2000);
}

// ============================================
// PHYSICS (Bubble Movement)
// ============================================

function updateBubblePositions() {
    tokens = tokens.map(token => {
        let newX = token.x + token.vx;
        let newY = token.y + token.vy;
        let newVx = token.vx;
        let newVy = token.vy;

        // Soft bounce
        if (newX > 1000 || newX < -1000) newVx = -newVx;
        if (newY > 600 || newY < -600) newVy = -newVy;

        return { ...token, x: newX, y: newY, vx: newVx, vy: newVy };
    });

    // Update DOM positions
    const bubbles = bubblesContainer.querySelectorAll(".bubble");
    bubbles.forEach(bubble => {
        const token = tokens.find(t => t.id === bubble.dataset.tokenId);
        if (token) {
            bubble.style.left = `${token.x}px`;
            bubble.style.top = `${token.y}px`;
        }
    });
}

// ============================================
// CANVAS INTERACTION (Zoom & Pan)
// ============================================

function setupCanvasInteraction() {
    const canvas = document.getElementById("canvas-container");

    // Wheel zoom
    canvas.addEventListener("wheel", (e) => {
        e.preventDefault();
        const zoomSpeed = 0.001;
        scale = Math.min(Math.max(0.1, scale - e.deltaY * zoomSpeed), 5);
        updateCanvasTransform();
    }, { passive: false });

    // Drag pan
    canvas.addEventListener("mousedown", (e) => {
        if (e.target === canvas || e.target.id === "grid-overlay") {
            isDragging = true;
            lastMousePos = { x: e.clientX, y: e.clientY };
        }
    });

    window.addEventListener("mousemove", (e) => {
        if (isDragging) {
            const dx = e.clientX - lastMousePos.x;
            const dy = e.clientY - lastMousePos.y;
            position.x += dx;
            position.y += dy;
            lastMousePos = { x: e.clientX, y: e.clientY };
            updateCanvasTransform();
        }
    });

    window.addEventListener("mouseup", () => {
        isDragging = false;
    });

    // Zoom buttons
    document.getElementById("zoom-in").addEventListener("click", () => {
        scale = Math.min(scale + 0.2, 5);
        updateCanvasTransform();
    });

    document.getElementById("zoom-out").addEventListener("click", () => {
        scale = Math.max(scale - 0.2, 0.1);
        updateCanvasTransform();
    });
}

// ============================================
// DATA POLLING
// ============================================

async function pollData() {
    const realTokens = await fetchRouterActivity();

    if (realTokens.length > 0) {
        usingRealData = true;
        statusText.textContent = "Live Basescan Feed";
        statusText.classList.add("live");
        pulseDot.classList.add("live");

        // Filter duplicates and add new tokens
        const existingIds = new Set(tokens.map(t => t.id));
        const newTokens = realTokens.filter(t => !existingIds.has(t.id));

        if (newTokens.length > 0) {
            tokens = [...tokens, ...newTokens].slice(-CONFIG.MAX_TOKENS);
            renderBubbles();
            renderFeed();
        }
    }
}

// ============================================
// INITIALIZATION
// ============================================

function init() {
    console.log("Basewump initialized - Monitoring Base chain Uniswap V2 Router");
    console.log("Router Address:", CONFIG.ROUTER_ADDRESS);

    // Setup canvas interaction
    setupCanvasInteraction();

    // Start physics loop
    setInterval(updateBubblePositions, 50);

    // Initial data fetch - real blockchain data only
    pollData();

    // Start polling every 5 seconds
    setInterval(pollData, CONFIG.POLL_INTERVAL);
}

// Start the app
document.addEventListener("DOMContentLoaded", init);
